//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Stage 1.0
import wesual.Ui    1.0

FocusScope {
    id : editor

    property alias font : width.font

    property SizeInfo size : Size.defaultSize()

    signal editFinished(SizeInfo size)

    function editAt(pos) {
        const widthBoxCenter = Qt.vector2d(width.x + width.width / 2,
                                           width.y + width.height/ 2);
        const heightBoxCenter = Qt.vector2d(height.x + height.width / 2,
                                            height.y + height.height/ 2);
        pos = Qt.vector2d(pos.x, pos.y);

        const widthDist = widthBoxCenter.minus(pos).length();
        const heightDist = heightBoxCenter.minus(pos).length();
        if (heightDist < widthDist) {
            height.setFocus();
        } else {
            width.setFocus();
        }
    }

    function tryCommit() {
        if (width.acceptableInput && height.acceptableInput) {
            const size = Size.sizeInfo(parseInt(width.text),
                                       parseInt(height.text));
            if (size !== editor.size) {
                editFinished(size);
            } else {
                editFinished(null);
            }
        } else {
            cancel();
        }
    }

    function cancel() {
        editFinished(null);
    }

    implicitWidth : content.implicitWidth
    implicitHeight : content.implicitHeight

    onActiveFocusChanged : {
        if (!activeFocus) {
            tryCommit();
        }
    }

    Keys.onEnterPressed : tryCommit()
    Keys.onReturnPressed : tryCommit()

    Row {
        id : content

        Shortcut {
            sequence : "Esc"
            onActivated : cancel()
        }

        TextInput {
            id : width

            function setFocus() {
                forceActiveFocus();
                cursorPosition = 0;
                selectAll();
            }

            activeFocusOnTab : true
            selectByMouse : true
            selectionColor : UiColors.getColor(UiColors.SelectionGreen)
            text : editor.size.width.toFixed(0)

            validator : IntValidator {
                bottom : 100
                top : 16384
            }

            Keys.onPressed : {
                if (event.key === Qt.Key_X) {
                    height.setFocus();
                    event.accepted = true;
                } else if (event.key === Qt.Key_Right) {
                    if (cursorPosition === text.length) {
                        height.forceActiveFocus();
                        height.cursorPosition = 0;
                        event.accepted = true;
                    } else {
                        event.accepted = false;
                    }
                }
            }

            Keys.onTabPressed : height.setFocus()

            MouseArea {
                anchors.fill : parent
                cursorShape : Qt.IBeamCursor
                acceptedButtons : Qt.NoButton
            }
        }

        Text {
            font : width.font
            text : "\u2009x\u2009"
        }

        TextInput {
            id : height

            function setFocus() {
                forceActiveFocus();
                cursorPosition = 0;
                selectAll();
            }

            activeFocusOnTab : true
            font : width.font
            selectionColor : UiColors.getColor(UiColors.SelectionGreen)
            selectByMouse : true
            text : editor.size.height.toFixed(0)

            validator : IntValidator {
                bottom : 100
                top : 16384
            }

            Keys.onPressed : {
                if (event.key === Qt.Key_X) {
                    width.setFocus();
                    event.accepted = true;
                } else if (event.key === Qt.Key_Left) {
                    if (cursorPosition === 0) {
                        width.forceActiveFocus();
                        width.cursorPosition = width.text.length;
                        event.accepted = true;
                    } else {
                        event.accepted = false;
                    }
                }
            }

            Keys.onBacktabPressed : width.setFocus()

            MouseArea {
                anchors.fill : parent
                cursorShape : Qt.IBeamCursor
                acceptedButtons : Qt.NoButton
            }
        }
    }
}
